<?php declare(strict_types=1);
/**
 * @author Ryan Spaeth <rspaeth@spaethtech.com>
 * @copyright 2025 - Spaeth Technologies, Archous Networks
 */

use DI\Bridge\Slim\Bridge;
use DI\ContainerBuilder;
use Slim\Routing\RouteCollectorProxy;
use Slim\Views\Twig;
use Slim\Views\TwigMiddleware;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Http\Message\ServerRequestInterface as Request;
use SpaethTech\UCRM\SDK\Middleware\QueryRoutingMiddleware;
use SpaethTech\UCRM\SDK\Controllers\ServicesCheckController;
use SpaethTech\UCRM\SDK\Controllers\ServicesSyncController;
use SpaethTech\UCRM\SDK\Controllers\ServicesWebhookController;
use SpaethTech\UCRM\SDK\Controllers\PluginController;
use SpaethTech\UCRM\SDK\Controllers\ApiErrorController;
use SpaethTech\UCRM\SDK\Controllers\FallbackController;

require_once __DIR__ . "/bootstrap.php";

$builder = new ContainerBuilder();
$builder->addDefinitions(__DIR__ . "/container.php");
$container = $builder->build();

$app = Bridge::create($container);

$app->add(TwigMiddleware::createFromContainer($app, Twig::class));

$app->addBodyParsingMiddleware();
$app->addRoutingMiddleware();

$errorMiddleware = $app->addErrorMiddleware(true, true, true);

$app->add(new QueryRoutingMiddleware());

// Auth middleware for X-Auth-Token validation
$authMiddleware = function (Request $request, $handler) use ($container) {
  error_log('[AUTH] Middleware called for: ' . $request->getUri()->getPath());

  $authHeader = $request->getHeaderLine('X-Auth-Token');
  error_log('[AUTH] Auth header value: ' . ($authHeader ?: 'EMPTY'));

  if (empty($authHeader)) {
    error_log('[AUTH] Missing auth header - returning 400');
    $response = new \Slim\Psr7\Response();
    $response->getBody()->write(json_encode(['error' => 'Missing X-Auth-Token header']));
    return $response->withStatus(400)->withHeader('Content-Type', 'application/json');
  }

  $plugin = $container->get(\SpaethTech\UCRM\SDK\Plugin::class);
  $expectedToken = $plugin->getAuth()->get('x_auth_token');
  error_log('[AUTH] Expected token: ' . substr($expectedToken, 0, 10) . '...');
  error_log('[AUTH] Received token: ' . substr($authHeader, 0, 10) . '...');

  if ($authHeader !== $expectedToken) {
    error_log('[AUTH] Invalid token - returning 401');
    $response = new \Slim\Psr7\Response();
    $response->getBody()->write(json_encode(['error' => 'Invalid X-Auth-Token']));
    return $response->withStatus(401)->withHeader('Content-Type', 'application/json');
  }

  error_log('[AUTH] Token valid - proceeding to handler');
  return $handler->handle($request);
};

$app->group("/api", function (RouteCollectorProxy $group) use ($authMiddleware) {

  $group->get(
    "/ping",
    function (Request $request, Response $response) {
      $response->getBody()->write(json_encode(["ping" => "pong"]));
      return $response->withHeader("Content-Type", "application/json");
    }
  );

  $group->get("/plugin", [PluginController::class, 'info'])->add($authMiddleware);

  $group->post("/services/check", [ServicesCheckController::class, 'check'])->add($authMiddleware);

  // No auth middleware for /services/sync
  $group->post("/services/sync", [ServicesSyncController::class, 'sync']);

  // Single service webhook endpoint - handles all webhook types
  $group->post("/services/webhook", [ServicesWebhookController::class, 'webhook'])->add($authMiddleware);

  $group->map(
    ["GET", "POST", "PUT", "DELETE", "PATCH", "HEAD", "OPTIONS"],
    "/{path:.*}",
    [ApiErrorController::class, 'notFound']
  )->add($authMiddleware);
});


// Define app routes
$app->get("/{path:.*}", [FallbackController::class, 'handlePublicRoute']);

// Run app
$app->run();
